/**@file camera.cpp: Windows capturing device function implementation. */

#include "../OsSupp.h"
#include "camera.h"
#include "../TimeTool.h"


extern int ComInitialized;	///< This variable is Windows specific from OsSupp.
#define HookCallbacks		///< Use callbacks, when this define is active

#ifndef N_
  #define N_(EnString) EnString
#endif

wxString TranslateVideoMode(const GUID & MediaSubType);
EXTERN_C const GUID MEDIASUBTYPE_A16R16G16B16;
EXTERN_C const GUID MEDIASUBTYPE_A16B16G16R16;
EXTERN_C const GUID MEDIASUBTYPE_A2B10G10R10;
EXTERN_C const GUID MEDIASUBTYPE_A2R10G10B10;
EXTERN_C const GUID MEDIASUBTYPE_I420;
EXTERN_C const GUID MEDIASUBTYPE_NV12;
EXTERN_C const GUID MEDIASUBTYPE_v210;
EXTERN_C const GUID MEDIASUBTYPE_R16G16B16;
EXTERN_C const GUID MEDIASUBTYPE_B16G16R16;
EXTERN_C const GUID MEDIASUBTYPE_Y422;
EXTERN_C const GUID MEDIASUBTYPE_R210;


EXTERN_C const GUID CLSID_KsDataTypeHandlerVideo2;


/////////////////////////////////////////////////////

CameraInput::CameraInput(const CameraInfo& ci): 
   Width(0), Height(0), FrameDuration(0), MinFrameDuration(0), MaxFrameDuration(0),
   GraphIsBuilt(false), m_cRef(1)
{
HRESULT hr;

  filter_base = NULL;
  IntName = NULL;
  pWxObj = NULL;

  // create the filter graph and necessary interfaces  
  graph.CreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC);
  if (graph==NULL) throw(WinAPIException(E_NOINTERFACE, N_("Failed to create filter graph")));
  IGraphBuilderPtr graphbuilder(graph);
  if (graphbuilder==NULL) throw(WinAPIException(E_NOINTERFACE, N_("Failed to get GraphBuilder interface for the filter graph")));

  // create the capture graph builder
  capbuilder.CreateInstance(CLSID_CaptureGraphBuilder2, NULL, CLSCTX_INPROC);
  if (capbuilder==NULL) throw(WinAPIException(E_NOINTERFACE, N_("Failed to create capture graph builder")));

  // attach the filter graph to the capture graph builder
  hr = capbuilder->SetFiltergraph(graphbuilder);
  if (FAILED(hr)) throw(WinAPIException(hr, N_("Failed to attach filter graph to the capture graph builder")));

  // instantiate the camera filter object  
  hr = ci.moniker->BindToObject(0, 0, IID_IBaseFilter, reinterpret_cast<void**>(&camera_base));
  if(FAILED(hr)) throw(WinAPIException(hr, N_("Failed to instantiate camera filter")));

  // add the camera filter to the filter graph  
  hr = graphbuilder->AddFilter(camera_base, L"Camera");
  if(FAILED(hr)) throw(WinAPIException(hr, N_("Failed to add camera filter to filter graph")));

    // create the sample grabber and necessary interfaces
  grabber.CreateInstance(CLSID_SampleGrabber, NULL, CLSCTX_INPROC);
  if(grabber==NULL) throw(WinAPIException(E_NOINTERFACE, N_("Failed to create sample grabber")));

   // Hook callbacks
#ifdef HookCallbacks    
  grabber->SetCallback(this,0);
#endif
  
  // add the sample grabber to the filter graph
  grabber_base = grabber;
  if(grabber_base==NULL) throw(WinAPIException(E_NOINTERFACE, N_("Failed to get BaseFilter interface for the sample grabber")));
  hr = graphbuilder->AddFilter(grabber_base, L"Grabber");
  if(FAILED(hr)) throw(WinAPIException(hr, N_("Failed to add sample grabber filter to filter graph")));

  // find the Capture pin on the camera
  hr = capbuilder->FindInterface(&PIN_CATEGORY_CAPTURE, &MEDIATYPE_Video, camera_base, IID_IAMStreamConfig, reinterpret_cast<void**>(&camera));
  if(FAILED(hr)) throw(WinAPIException(E_NOINTERFACE, N_("Failed to get the IAMStreamConfig the camera output pin")));

  // render the graph stream
  //     - removed from here and moved to the method Start();  

  //SelectVideoMode(); don't select video mode just here, select it from application

  // store real camera name, do it as a last action, otherwise leak occurs during exception.
  std::wstring str = ci.GetName();
  if(str.length()>0)
    {
    IntName = (char *)calloc(str.length()+1,sizeof(char));
    if(IntName)
      for(unsigned i=0; i<str.length(); i++)
        {
        IntName[i] = (char)str[i];
        }
    }
}


CameraInput::~CameraInput()
{
  if(IntName) {free(IntName);IntName=NULL;}
  pWxObj = NULL;
}


/** Start grabbing and build a chain when needed. */
bool CameraInput::Start(void)
{
  if(!BuildGraph()) return false;    

  grabber->SetBufferSamples(TRUE);    
  graph->Run();
  return true;
}


/** Stops grabbing and optionally break a chain inside chart. 
 *  @param[in] StopGraph - optionally break a capture graph. Default value 'false'.
 *  @return Exception _com_error when COM fails. */
void CameraInput::Stop(bool StopGraph)
{
  if(!GraphIsBuilt) return;	//if a graph is not built, it make no sense to stop it

  grabber->SetBufferSamples(FALSE);
  graph->StopWhenReady();

  if(StopGraph)
    {
    HRESULT hr;

    IFilterGraphPtr ifg(graph);
    hr=ifg->RemoveFilter(grabber_base);		//remove all components from a chart
    hr=ifg->RemoveFilter(camera_base);		//   this breaks a chain as a side effect
    if(filter_base!=NULL)
      hr=ifg->RemoveFilter(filter_base);

    IGraphBuilderPtr graphbuilder(graph);	//add them again and don't build chain
    hr = graphbuilder->AddFilter(camera_base, L"Camera");
    hr = graphbuilder->AddFilter(grabber_base, L"Grabber");
    if(filter_base!=NULL)
      hr = graphbuilder->AddFilter(filter_base, L"Filter");
        
    GraphIsBuilt=false;
    }
}


/** Build direct view graph. A graph is fully build only before capturing, otherwise
 *    it blocks device.
 *  @return Exception _com_error when COM fails. */
bool CameraInput::BuildGraph(void)
{
HRESULT hr;

  if(GraphIsBuilt) return true;  

  hr = capbuilder->RenderStream(&PIN_CATEGORY_CAPTURE, &MEDIATYPE_Video, camera_base, filter_base, grabber_base);  

	// This piece of code provides same functionality as "RenderStream" for diagnostic purpose.
  if(FAILED(hr))
  {     
    IPin *CamPIN;
    IPin *GrabPIN;
    IEnumPins *pEnum;

    hr = camera_base->EnumPins(&pEnum);
    hr = pEnum->Next(1,&CamPIN,NULL);
    pEnum->Release(); 

    hr = grabber_base->EnumPins(&pEnum);
    hr = pEnum->Next(1,&GrabPIN,NULL);  
    pEnum->Release();

    IGraphBuilderPtr graphbuilder(graph);
    hr = graphbuilder->Connect(GrabPIN,CamPIN);
    //hr = graphbuilder->Render(GrabPIN); - not neccessary

    CamPIN->Release();
    GrabPIN->Release();
  }

  if(FAILED(hr)) return false;
	  //throw(WinAPIException(hr, "The capture device may already be in use by another application."));

  GraphIsBuilt = true;
return true;
}


void CameraInput::UseFilter(void)
{
  if(filter_base==NULL)
    {
    try {
        filter_base.CreateInstance(CLSID_DVVideoCodec, NULL, CLSCTX_INPROC);// Create the DV Decoder (CLSID_DVVideoCodec)
        }
    catch(...) {filter_base=NULL;}
    if(filter_base!=NULL)
        {
        if(GraphIsBuilt) Stop(true);
        IGraphBuilderPtr graphbuilder(graph);
        graphbuilder->AddFilter(filter_base, L"Filter");
        }
    } 
}


void CameraInput::UseFilterMJPG(void)
{
  if(filter_base==NULL)
    {
    try {
        filter_base.CreateInstance(CLSID_MjpegDec, NULL, CLSCTX_INPROC);// Create the DV Decoder (CLSID_DVVideoCodec)
        }
    catch(...) {filter_base=NULL;}
    if(filter_base!=NULL)
        {
        if(GraphIsBuilt) Stop(true);
        IGraphBuilderPtr graphbuilder(graph);
        graphbuilder->AddFilter(filter_base, L"Filter");
        }
    } 
}


/** Read currently selected frame rate, minimal and maximal available frame rate. 
 *  @param[in] Specify which value to read, default 0. */
double CameraInput::GetFrameRate(int What)
{
switch(What)
  {
  case 1: if(MinFrameDuration<=0) return 0;
	  return 10000000/(double)MinFrameDuration;
  case 2: if(MaxFrameDuration<=0) return 0;
	  return 10000000/(double)MaxFrameDuration;
  default:if(FrameDuration<=0) return 0;
          return 10000000/(double)FrameDuration;
  }
}


/** Get friendly camera name. */
char *CameraInput::GetName(void) const
{
  if(IntName) return IntName;
return "";
}


/** Scan camera capabilities and select best video mode. 
  * @return Exception _com_error when COM fails. */
void CameraInput::SelectVideoMode(int ReqWidth, int ReqHeight, int ReqDepth, double NewFrameRate)
{
TAudVidConfCaps *PAudVidConfCaps;

  if(camera==NULL) return;
  if(!BuildGraph()) return;
 
  int count = sizeof(AUDIO_STREAM_CONFIG_CAPS);
  int size = sizeof(VIDEO_STREAM_CONFIG_CAPS);
  camera->GetNumberOfCapabilities(&count, &size);

  AM_MEDIA_TYPE* mt=NULL;
  VIDEOINFOHEADER *pVIH;
  int planes;

  int BestI=0;	
  double qualityI=-1000;
  double quality=-10000;   ///< negative quality means unsatisfied contraint, 0 is full satisfaction, positive denotes bigger resolution

  PAudVidConfCaps = (TAudVidConfCaps *)CoTaskMemAlloc(sizeof(TAudVidConfCaps));
  if(PAudVidConfCaps==NULL) return;

  for(int i=0; i<count; i++)		// traverse all available resolutions
  {
    mt = NULL;
    if(FAILED(camera->GetStreamCaps(i, &mt, (BYTE*)PAudVidConfCaps)))
        continue;
    if(mt==NULL) continue;   

    pVIH = (VIDEOINFOHEADER*)mt->pbFormat;

    planes = (8*mt->lSampleSize)/(PAudVidConfCaps->v.InputSize.cx*labs(PAudVidConfCaps->v.InputSize.cy));
    if(planes<8)
    {
      planes = pVIH->bmiHeader.biPlanes*pVIH->bmiHeader.biBitCount;
    }

    if(ReqWidth>0 && ReqHeight!=0)
    {
      if(labs(PAudVidConfCaps->v.InputSize.cy)>=labs(ReqHeight))
      {
        if(PAudVidConfCaps->v.InputSize.cx>=ReqWidth)
        {
          quality = labs(labs(PAudVidConfCaps->v.InputSize.cy)-labs(ReqHeight)) +
                    labs(PAudVidConfCaps->v.InputSize.cx-ReqWidth);
          if(planes!=ReqDepth)
	    switch(planes)
	    {
	         case 32: quality+=11; break;
                 case 24: quality+=10; break;
                 case 12: quality+=5; break;
                 case 8: quality+=1; break;
                 default: quality+=30;		//unknown depth
	    }
          }
          else
             quality = PAudVidConfCaps->v.InputSize.cx-ReqWidth;   //not satisfied width
        }
        else
        {
          if(PAudVidConfCaps->v.InputSize.cx>=ReqWidth)
              quality = labs(labs(PAudVidConfCaps->v.InputSize.cy)-labs(ReqHeight));  //not satisfied height
          else
              quality = labs(labs(PAudVidConfCaps->v.InputSize.cy)-labs(ReqHeight)) +
                        labs(PAudVidConfCaps->v.InputSize.cx-ReqWidth);   //not satisfied both
        }

        wxString str = TranslateVideoMode(mt->subtype);
        if(!str.empty())
        {
          if(str[0]=='{') quality+=50;		//unknown format
          if(str[0]=='!') quality+=50;		//unsupportedn format
        }
    
        if(mt->formattype==CLSID_KsDataTypeHandlerVideo2)
          quality+=80;    
      }
         

      if((quality>=0 && (quality<qualityI || qualityI<0) )  ||
         (quality<0 && quality>qualityI) )
	{
	qualityI=quality;
	BestI=i;
	}

      if(mt->pbFormat != NULL)		// There was a leak!
        {
	CoTaskMemFree(mt->pbFormat);
	mt->pbFormat = NULL;
        }
      CoTaskMemFree(mt); mt=NULL;
    }

	//setup a best format found;   
   if(SUCCEEDED(camera->GetStreamCaps(BestI, &mt, (BYTE*)PAudVidConfCaps)))
     if(mt!=NULL)
     {
     pVIH = (VIDEOINFOHEADER*)mt->pbFormat;

     FrameDuration = pVIH->AvgTimePerFrame;
     MinFrameDuration = PAudVidConfCaps->v.MinFrameInterval;
     MaxFrameDuration = PAudVidConfCaps->v.MaxFrameInterval;

     if(NewFrameRate>0) pVIH->AvgTimePerFrame = (REFERENCE_TIME)(10000000/NewFrameRate);
     VideoFormat = vfUnspecified;

     camera->SetFormat(mt);

     if(mt->subtype==MEDIASUBTYPE_RGB565) VideoFormat=vfRGB565;
     if(mt->subtype==MEDIASUBTYPE_RGB555) VideoFormat=vfRGB555;
     if(mt->subtype==MEDIASUBTYPE_NV12) VideoFormat=vfNV12;
     if(mt->subtype==MEDIASUBTYPE_v210) VideoFormat=vfV210;
     if(mt->subtype==MEDIASUBTYPE_R210) VideoFormat=vfR210;
     if(mt->subtype==MEDIASUBTYPE_A2B10G10R10) VideoFormat=vfA2B10G10R10;
     if(mt->subtype==MEDIASUBTYPE_A2R10G10B10) VideoFormat=vfA2R10G10B10;
     if(mt->subtype==MEDIASUBTYPE_R16G16B16) VideoFormat=vfR16G16B16;
     if(mt->subtype==MEDIASUBTYPE_B16G16R16) VideoFormat=vfB16G16R16;
     if(mt->subtype==MEDIASUBTYPE_A16R16G16B16) VideoFormat=vfA16R16G16B16;
     if(mt->subtype==MEDIASUBTYPE_A16B16G16R16) VideoFormat=vfA16B16G16R16;
     if(mt->subtype==MEDIASUBTYPE_Y422)	VideoFormat=vfY422;	// Known as UYVY, Y422 or UYNV: https://wiki.videolan.org/YUV
     if(mt->subtype==MEDIASUBTYPE_UYVY) VideoFormat=vfUYVY;
     if(mt->subtype==MEDIASUBTYPE_YVYU) VideoFormat=vfYVYU;
     if(mt->subtype==MEDIASUBTYPE_YUYV) VideoFormat=vfYUYV;
     if(mt->subtype==MEDIASUBTYPE_YUY2) VideoFormat=vfYUYV;
     if(mt->subtype==MEDIASUBTYPE_YV12) VideoFormat=vfI420;	// YUV 4:2:0 (I420/J420/YV12)
     if(mt->subtype==MEDIASUBTYPE_I420) VideoFormat=vfI420;
     if(mt->subtype==MEDIASUBTYPE_RGB8)
	{
	VideoFormat = vfRGB8;	    
	if(mt->pbFormat!=NULL && mt->cbFormat>=sizeof(VIDEOINFO))
	  {
	  VIDEOINFO *pvi = (VIDEOINFO*)mt->pbFormat;
	  if(pCameraPalette==NULL) pCameraPalette=(TCameraPalette*)malloc(sizeof(TCameraPalette)*256);
	  if(pCameraPalette != NULL)
	    {
	    for(int i=0; i<256; i++)
	      {
	      pCameraPalette[i].R = pvi->TrueColorInfo.bmiColors[i].rgbRed;
	      pCameraPalette[i].G = pvi->TrueColorInfo.bmiColors[i].rgbGreen;
	      pCameraPalette[i].B = pvi->TrueColorInfo.bmiColors[i].rgbBlue;
	      }
	    }
	  }	    
	}

     if(mt->subtype==MEDIASUBTYPE_dvsl || mt->subtype==MEDIASUBTYPE_dvsd
         || mt->subtype==MEDIASUBTYPE_dvhd 
         //mt->subtype==MEDIASUBTYPE_dv25 || mt->subtype==MEDIASUBTYPE_dv50 ||
         //mt->subtype==MEDIASUBTYPE_dvh1   //proffessional types, CLSID not found.
        )
	{  
	UseFilter();
        mt->subtype = MEDIASUBTYPE_RGB24;
	grabber->SetMediaType(mt);
	}

     if(mt->subtype==MEDIASUBTYPE_MJPG)
       {
       UseFilterMJPG();
       mt->subtype = MEDIASUBTYPE_RGB24;
       grabber->SetMediaType(mt);
       }

     Width = PAudVidConfCaps->v.InputSize.cx;
     Height = PAudVidConfCaps->v.InputSize.cy;
     Planes=(8*mt->lSampleSize)/(PAudVidConfCaps->v.InputSize.cx*labs(PAudVidConfCaps->v.InputSize.cy));
     if(Planes<8)
     {
       Planes = pVIH->bmiHeader.biPlanes*pVIH->bmiHeader.biBitCount;
     }

	/* Read a Frame rate actually set. */
     if(NewFrameRate>0)
       {
       if(mt->pbFormat != NULL)		// There was a leak!
          {
	  CoTaskMemFree(mt->pbFormat);
	  mt->pbFormat = NULL;
          }
       CoTaskMemFree(mt); mt=NULL;

       if(SUCCEEDED(camera->GetFormat(&mt)))
         if(mt!=NULL)
           {
           pVIH = (VIDEOINFOHEADER*)mt->pbFormat;
           FrameDuration = pVIH->AvgTimePerFrame;         
         }
       }
     }

   if(PAudVidConfCaps) {CoTaskMemFree(PAudVidConfCaps);PAudVidConfCaps=NULL;}
   if(mt) 
   {
     if(mt->pbFormat != NULL)		// There was a leak!
        {
	CoTaskMemFree(mt->pbFormat);
	mt->pbFormat = NULL;
        }
     CoTaskMemFree(mt);mt=NULL;
   }
}


unsigned char *CameraInput::GetBufferData(void)
{
  if(rawbuffer.size()<=0) return NULL;
  return &rawbuffer.front();
}


/** Display a generic property page that is camera dependent. */
void CameraInput::ShowPropertyPage(size_t hOwner)
{
    // Get the list of property pages to display
  ISpecifyPropertyPages* piPages;

  if(camera_base==NULL) return;
  HRESULT hResult = camera_base->QueryInterface(__uuidof(piPages), reinterpret_cast<void**>(&piPages));
  if(SUCCEEDED(hResult))
  {
    CAUUID Pages;
    hResult = piPages->GetPages(&Pages);

    if(SUCCEEDED(hResult) && Pages.cElems > 0)
    {
      IUnknown *pIU = (IUnknown*)camera_base;
      hResult = OleCreatePropertyFrame(
                            (HWND)hOwner,	//m_hWnd,
                            0,
                            0,
                            L"Camera Properties",
                            1,
                            &pIU,
                            Pages.cElems,
                            Pages.pElems,
                            LANG_USER_DEFAULT,
                            0,
                            NULL);

      CoTaskMemFree(Pages.pElems);
    }
    piPages->Release();
  }
}


/** Scan one image from camera device 
  * @return Exception _com_error when COM fails. 
            Exception WinAPIException on bad HRESULT. */
void CameraInput::GrabImage(void)
{
    // ensure there's enough room in the image buffer
  AM_MEDIA_TYPE mediatype;
  mediatype.pbFormat = NULL;
  if(FAILED(grabber->GetConnectedMediaType(&mediatype)))
    {
    Planes = Height = Width = 0; //media type cannot be read	
    rawbuffer.resize(0);	//  - so sampe size doesn't make sense
    return;
    }

  rawbuffer.resize(max(mediatype.lSampleSize,Width*labs(Height)*Planes/8));

    // grab the image
  long bufsize = rawbuffer.size();
  HRESULT hr = grabber->GetCurrentBuffer(&bufsize, reinterpret_cast<long*>(&rawbuffer.front()));

  if(mediatype.pbFormat)
  {
    CoTaskMemFree(mediatype.pbFormat);	// The caller must free the format block of the media type. 
  }

  if (FAILED(hr)) throw(WinAPIException(hr, "Failed to read an image from the grabber"));
  //rawbuffer.resize(bufsize);
}

//////////// Interface IUnknown - procedures////////////
ULONG CameraInput::AddRef()
{
  return(InterlockedIncrement(&m_cRef));
}


ULONG CameraInput::Release()
{
 if(InterlockedDecrement(&m_cRef)!=0)
 		return(m_cRef);
// delete(this); - this object should not be deleted from COMs
 m_cRef=0;
 return(0);
}


HRESULT CameraInput::QueryInterface(REFIID riid, void **ppv)
{
  if(riid==IID_IUnknown) 
	{
	*ppv=(IUnknown*)(CameraInput*)(this);
	}
  else if(riid==IID_ISampleGrabberCB) 
	*ppv=(CameraInput*)this;
  else 	{
	*ppv=NULL;
	return E_NOINTERFACE;
	}

  AddRef();

  return(S_OK);
}



// Interface ISampleGrabberCB - procedure prototypes
HRESULT __stdcall CameraInput::BufferCB(double /*SampleTime*/, BYTE * /*pBuffer*/, long /*BufferLen*/)
{
  return E_NOTIMPL;
}


HRESULT __stdcall CameraInput::SampleCB(double SampleTime, IMediaSample *pSample)
{
  TickCount = GetTickCount_ms();
  FrameCounter++;

#ifdef HookCallbacks
  try 
    {FireSampleCB(pWxObj);}
  catch(...)
    {}
#endif

  return S_OK;
}



///////////////////////////////////////////////////////

/** Empty camera placeholder for not working cameras. */
class NoCamera: public CameraDevice
{
public: 
  virtual char *GetName(void) const {return "NOT WORKING!";}
};


///////////////////////////////////////////////////////


/** Class factory for Logitech Quick Cam. */
class LogitechQCamManagerDriver: public BazeCameraDriver
{
protected:  
  std::vector<CameraInfo> camerainfo;   ///< Video input device monikers, used for drivers object build
  std::vector<CameraDevice *>drivers;   ///< List of all active driver objects

  void ScanForDevices(void);		///< Get device list
  void Erase(void);			///< Erase all devices
  bool DCOMsInit;			///< DCOM's initialisation flag

public:
  virtual ~LogitechQCamManagerDriver();
  LogitechQCamManagerDriver(void): DCOMsInit(false) {};

  virtual char *GetName(void) const {return "Camera:";}
  virtual CameraDevice *GetDriver(int Index, const char *Name=NULL);
};


/** Deallocate all drivers from destructor and optionally shut down DCOMs */
LogitechQCamManagerDriver::~LogitechQCamManagerDriver()
{  
  Erase();
  if(DCOMsInit)
    {
    DoneDCOM();
    DCOMsInit=false;
    }
}


/** Get working object of camera driver on n'th position. 
      - no exception should be passed during normal behaviour.
 *  @param[in] Index index of camera driver
 *  @param[in] Name  Name of driver, currently unused
 *  @return CameraDevice object on given index
	    NoCamera     Placeholder object when camera exists, but don't work
	    NULL         Index is out of camera list, or camera list is empty.
      - no exception should be passed during normal behaviour.  */
CameraDevice *LogitechQCamManagerDriver::GetDriver(int Index, const char *Name)
{
  if(Index==ERASE_ALL)
    {
    Erase();
    return NULL;
    }
  if(camerainfo.size()<=0) 
    {
    try {ScanForDevices();}	//no device has been found - find any now
    catch(...) {return(NULL);}
    }

  if((int)camerainfo.size()<=Index) return NULL;

  if(Index>=(int)drivers.size())
    drivers.resize(Index+1);  

  if(drivers[Index]!=NULL) return drivers[Index];
  
  try {
      CameraInput *CDx;
      CDx = new CameraInput(camerainfo[Index]);
      drivers[Index] = CDx;
      }
  catch(...) 
      {drivers[Index] = new NoCamera();}	// when constructor raises exception, create camera placeholder
  
  return drivers[Index];
}


/** Cleanup whole list of cameras and delete all camera objects. */
void LogitechQCamManagerDriver::Erase(void)
{
  for(std::vector<CameraDevice *>::iterator i=drivers.begin(); i!=drivers.end(); ++i)
    {
    if(*i)
      {
      delete *i;	//erase cached driver
      *i = NULL;
      }      
    } 
  drivers.clear();
  camerainfo.clear();
}


/** Get a list of all cameras available. 
   @return WinAPIException on failure. */
void LogitechQCamManagerDriver::ScanForDevices(void)
{
HRESULT hr;

  if(!DCOMsInit) 
    {
    InitDCOM();
    DCOMsInit=true;
    }

	// destroy the existing device entries
  camerainfo.clear();

  // create the system device enumerator
  _COM_SMARTPTR_TYPEDEF(ICreateDevEnum, __uuidof(ICreateDevEnum));
  try
    {
    ICreateDevEnumPtr devenum(CLSID_SystemDeviceEnum, NULL, CLSCTX_INPROC);
    if (devenum==NULL) throw(WinAPIException(E_NOINTERFACE, N_("Failed to create system enumerator")));

      // create an enumerator for the video capture devices
    IEnumMonikerPtr classenum;
    hr = devenum->CreateClassEnumerator(CLSID_VideoInputDeviceCategory, &classenum, 0);
    if (FAILED(hr)) throw(WinAPIException(hr, N_("Failed to create video input class enumerator")));

      // return if there are no enumerators - this means no camera devices
    if (classenum == NULL) return;

      // enumerate the video input device monikers.
    IMonikerPtr moniker;
    ULONG numfetched;
    while(classenum->Next(1, &moniker, &numfetched) == S_OK)
      {
      IPropertyBagPtr pbag;
      hr = moniker->BindToStorage(0, 0, IID_IPropertyBag, reinterpret_cast<void**>(&pbag));
      if (FAILED(hr)) continue;

      // retrieve the camera filter's friendly name
      _variant_t name;
      hr = pbag->Read(L"FriendlyName", &name, 0);
      if (SUCCEEDED(hr))
        {
        camerainfo.push_back(CameraInfo(_bstr_t(name), moniker));
        }
      }
   }
 catch(...) 
   {
   throw(WinAPIException(E_NOINTERFACE, N_("Occured exception during camera enumeration")));
   }  

}


/** static instance that generates drivers
 *  Drivers are not generated from constructor, they are created during first 
 *  call 'ScanForDevices()'. */
LogitechQCamManagerDriver LogitechQCam;	

