/******************************************************************************
 * program:     rasimg library                                                *
 * function:    Library for saving/loading special ftg format.                *
 * modul:       ras_ftg.cc                                                    *
 * licency:     GPL or LGPL                                                   *
 ******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "typedfs.h"
#include "image.h"
#include "struct.h"

#include "imgsupp.h"


#ifdef SupportFTG

#include "bitstrea.h"


static const char FTG_IDENTIF[4] = {'F','&','T','g'};


//int SavePictureBMP(const char *Name,const Image &Img); ///!!!


//////!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
/** Peels one bit plane.
 * @param[in]	obr	Source image with multiple planes.
 * @param[out]	PlaneX	Preallocated 1 plane image.
 * @param[in]	PlaneNum	Number of a plane. */
static void PeelPlane(Raster2DAbstract & obr, Raster2DAbstract & PlaneX, int PlaneNum)
{
unsigned y, MaxY;

 if(PlaneX.GetPlanes()!=1 || PlaneX.Size1D!=obr.Size1D) return;

 if(PlaneNum<0 || PlaneNum>=obr.GetPlanes() || obr.Data2D==NULL) return;
 //PlaneX.Create(obr.SizeX(),obr.SizeY(),1 | NoFill);

 MaxY=(obr.Size2D<PlaneX.Size2D)?obr.Size2D:PlaneX.Size2D;
 for(y=0; y<MaxY; y++)		// main row loop
 {
   obr.GetRowRaster(y)->Peel1Bit(PlaneX.GetRowRaster(y)->Data1D,PlaneNum);
 }
}


/////////////////////////////////////////////////////////////
static void JoinPlane(Raster2DAbstract & PlaneX, Raster2DAbstract & Obr, uint8_t PlaneNum)
{
unsigned y, MaxY;

 if(((PlaneNum<0)||(PlaneNum>=Obr.GetPlanes()))||(PlaneX.Data2D==NULL)) return;
 if(Obr.Data2D==NULL)
    {/*	//if no raster available, create it and hope that PlaneNum is a maximal plane
    *Obr=CreateRaster2D(PlaneX.Size1D,PlaneX.Size2D,PlaneNum);
    if(Obr.Data2D==NULL)*/
	return;
    }

 MaxY=(Obr.Size2D<PlaneX.Size2D)?Obr.Size2D:PlaneX.Size2D;
 for(y=0;y<MaxY;y++)		// hlavni radkova smycka
    {
    Obr.GetRowRaster(y)->Join1Bit(PlaneX.GetRowRaster(y)->Data1D,PlaneNum);
    }
}
//////!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


//--------------------------FTG---------------------------
#ifdef SupportFTG

#include "xlat_swp.h"


//This is file type identifier. It must be present on the begining of each file.
typedef struct
	{
	char Identif[4];
        uint16_t version;
	} ImgHeader;


///This is object descriptor. Each file consists of set of this objects.
/*{Definovan objekty:0 - przdn? objekt
		    1 - textov? koment
		    2 - zvukov? koment	xxx never implemented
		    3 - ikona			xxx never implemented
		    4 - raster image
		    5 - velk? obrzek (rastrov?) xxx never implemented
		    6 - ikona (vektorov)	xxx never implemented
		    7 - obrzek (vektorov?)	xxx never implemented
		    8 - velk? obrzek (vektorov?) xxx never implemented

Podtypy:	   20 - palete
		   21 - souadnice pesnho um?st?n? xxx never implemented
		   22 - pomocn informace	xxx never implemented

		   40 - EXIF profile
		   41 - XMP profile
		   42 - ICC profile

Speciln? typy	  100 - kontejner (adres)} xxx never implemented */
typedef struct 
        {
	uint8_t    type;
        uint32_t   size;
	} ImgObject;

#define SizeImgObject 5

inline long LoadFtgObject(FILE *f, ImgObject &SU)
{
#if defined(__PackedStructures__)
 return(fread(&SU,1,sizeof(SU),f));
#else
 return(loadstruct(f,"bd", &SU.type, &SU.size));
#endif
}

inline long SaveFtgObject(FILE *f, ImgObject &SU)
{
#if defined(__PackedStructures__)
 return(fwrite(&SU,1,sizeof(SU),f));
#else
 return(savestruct(f,"bd", SU.type, SU.size));
#endif
}


typedef enum
{
	Inside_Gray	=  0,
	Inside_RGB	= 10,
	Inside_Palette	= 20
} E_INTERPRETATION;


//Here follows definitions for each file object record
#define SIZE_RAS_DATA	7
typedef struct
	{
	uint16_t   SizeX;
	uint16_t   SizeY;
	uint8_t    planes;
	uint8_t    interpretation;
	uint8_t    compression;
	} RasterData;

typedef struct
	{
	uint16_t	items;
	uint8_t		typ;            // 2-word Packed; 3-24 bits; 6-word for R,G,B
	uint16_t	startItem;
	uint8_t		destPlanes;
	} PaletteData;


inline long LoadFtgData(FILE *f, RasterData &SU)
{
#if defined(__PackedStructures__)
 return(fread(&SU,1,sizeof(SU),f));
#else
 return(loadstruct(f,"wwbbb",
	 &SU.SizeX, &SU.SizeY,
	 &SU.planes, &SU.interpretation, &SU.compression));
#endif
}

inline long SaveFtgData(FILE *f, RasterData &SU)
{
#if defined(__PackedStructures__)
 return(fwrite(&SU,1,sizeof(SU),f));
#else
 return(savestruct(f,"wwbbb",
	 SU.SizeX, SU.SizeY,
	 SU.planes, SU.interpretation, SU.compression));
#endif
}


//This structures are used for data handling.
class TImageStruct;

class TObjectReader             //reads all objects stored in file
        {
public: FILE *ImgFile;
        long StartPosition;   //if ==0 then object contain invalid data
        long NextObjPos;
        long ActualPosition;
        long LastObjPosition;         //dodelat!!!!!
        ImgObject ActualObj;

        TImageStruct *AssignedImg;

        TObjectReader(void);
        void Init(long Position);
        void Rewind(void);
        int ReadObject(long Position);
	};

class TImageStruct              //main file descriptor for handling file
        {
public: FILE *ImgFile;
        uint16_t version;

        TObjectReader RootReader;

        int AttachFile(const char *FileName);
        void DeattachFile(void);

        TImageStruct();
        ~TImageStruct(void);
	};

/* Methods of class TObjectReader */
TObjectReader::TObjectReader(void) 
{
  AssignedImg=NULL;
  memset(&StartPosition,0,sizeof(StartPosition));
  return;
}


//This call initialize all object. You can read objects only after this call.
//      you may set Position=0 for actual file position
void TObjectReader::Init(long Position)
{
 StartPosition=0;
 if(AssignedImg==NULL) return;
 if(AssignedImg->ImgFile==NULL) return;
 if(Position!=0) StartPosition=Position;
	    else StartPosition=ftell(AssignedImg->ImgFile);
 NextObjPos=ActualPosition=StartPosition;
}

void TObjectReader::Rewind(void)
{
 NextObjPos=StartPosition;
}


int TObjectReader::ReadObject(long Position)
{
 if(StartPosition==0) return(-11);      //error - Object not initialized
 if(NextObjPos==0) return(-12);         //error - Object not initialized

 if(Position!=0) ActualPosition=Position; //correct actual position if it is known
 if(ActualPosition!=NextObjPos)
        {
        if(fseek(AssignedImg->ImgFile,NextObjPos,SEEK_SET)!=0) return(-12);
        ActualPosition=NextObjPos;
        }
 if(feof(AssignedImg->ImgFile)) return(-13);

 NextObjPos=0;

 if(LoadFtgObject(AssignedImg->ImgFile,ActualObj)!=SizeImgObject)
	return(-12);    //error - cannot read object descriptor;

 ActualPosition+=SizeImgObject;
 NextObjPos = ActualPosition+ActualObj.size;
 return(ActualObj.type);
}



/* Methods of class TImageStruct */
inline TImageStruct::TImageStruct(void)
{version=0;ImgFile=NULL;RootReader.AssignedImg=this;return;}

inline TImageStruct::~TImageStruct(void)
{DeattachFile();}


inline long LoadFtgHeader(FILE *f, const ImgHeader &HDR)
{
#if defined(__PackedStructures__)
 return(fread(&HDR,1,sizeof(HDR),f));
#else
 return(loadstruct(f,"a4w",
	 HDR.Identif, &HDR.version));
#endif
}


int TImageStruct::AttachFile(const char *FileName)
{
ImgHeader Hdr;

 if(ImgFile!=NULL) fclose(ImgFile);
 ImgFile=fopen(FileName,"rb");
 if(ImgFile==NULL) return(ErrOpenFile);         //File cannot be opened

 if(LoadFtgHeader(ImgFile,Hdr)!=6) return -3; 

 if(memcmp(Hdr.Identif,FTG_IDENTIF,4))
        {
        DeattachFile();
        return(-2);                     //Another file
        }
 if(Hdr.version>=0x200)
        {
//      printf("version %X",Hdr.version);
        DeattachFile();
        return(-3);                     //Unsupported version
        }
 version=Hdr.version;

 RootReader.Init(sizeof(Hdr));

 return(0);                             //File opened OK
}

void TImageStruct::DeattachFile(void)
{
 version=0;
 if (ImgFile==NULL) return;
 fclose(ImgFile);
 ImgFile=NULL;
}

/*------------Procedures for F&Tg compression/decompression------------------*/
#define  SchMask  0x7
#define   Schless_2D    1
#define   Schless_3D    2
#define   Schless_3DVar 3
#define  HufMask   0x18
#define  Huff         8
#define  HuffVar   0x10
#define  RemapPal  0x20
#define  GrayCode  0x40
#define FtGrCompression		(Schless_3DVar+HuffVar)		//actual type of compression

#define CBufSize 2048		//Size of bitstream file cache. Must be >=4!

#define ProbeCells   10

typedef uint8_t bHuf[MAX_CODES];
typedef uint8_t TSmaskFull[256];


int8_t Repeaters[8]  = {0,0,3,7,15,31,63,127};
int8_t ZeroValues[8] = {0,0,2,6,14,30,62,126};
int8_t Aditors[8]    = {0,0,1,3, 7,15,31, 63};


inline int GetBinPix(unsigned x, uint8_t *b)
{
 b+=x >> 3;
//GetBinPix:=b^ and ($80 >>(x and $7));{}
 return(*b & (1 <<(x & 7)));
}

//This procedure set one bit in the bit array
inline void SetBinPix(uint16_t x,uint8_t *b)
{
 b+=x >> 3;
//{ b^:=b^ or ($80 >>(x and $7));{}
 *b=*b | (1 << (x & 7));
}


/*This procedure return a size of the number coded in log growth code*/
unsigned SizeNum(unsigned n)
{
 if(n<=2)   return(2);
 if(n<=4)   return(3);
 if(n<=8)   return(5);
 if(n<=16)  return(7);
 if(n<=32)  return(9);
 if(n<=64)  return(11);
 if(n<=128) return(13);
 if(n<=256) return(15);
 if(n<=512) return(17);
 return(19);
}


//Selecting neighbourhood planes for RGB
void Setn3n4forRGB(Raster2DAbstract & obr,uint8_t n,uint8_t & n2plane,uint8_t & n3plane,uint8_t & n4plane)
{
//  position	  ( 1, 2, 3, 4, 5, 6, 7, 8,   9,10,11,12,13,13,15,16,  17,18,19,20,21,22,23,24);
const uint8_t Re2[24]={ 2, 3, 4, 5, 6, 7, 8,16,  10,11,12,13,14,15,16,24,  18,19,20,21,22,23,24,255};
const uint8_t Re3[24]={ 9,10,11,12,13,14,15,15,  11,12,13,14,15,16,22,23,  19,20,21,22,23,24,255,255};
const uint8_t Re4[24]={17,18,19,20,21,22,23,24,  17,18,19,20,21,22,23,22,  20,21,22,23,24,255,255,255};

if ((n>obr.GetPlanes())||(n<=0)) return;
n2plane=n+1;
n3plane=n+2;
n4plane=n+3;
if((FtGrCompression & RemapPal) != 0)
    {
    if(obr.GetPlanes()==24)		//True Color RGB mode}
	{
	n2plane=Re2[n];
	n3plane=Re3[n];
	n4plane=Re4[n];
	}
   }
if(--n2plane>=obr.GetPlanes()) n2plane=255;
if(--n3plane>=obr.GetPlanes()) n3plane=255;
if(--n4plane>=obr.GetPlanes()) n4plane=255;
}

//This procedure decompress packed description of the estimator
void ExplodeBitFlags(uint16_t & numera,TSmaskFull & SmaskF,TSmaskFull & SmaskEx)
{
uint16_t ri,i;

  memset(SmaskEx,0,sizeof(SmaskF));
  for(i=0;i<=2047;i++)
	{
	ri=i;
	if((numera & 2048) != 0)  ri= ((ri & 0xF000) >> 1) | (ri & 0x7FF);
	if((numera & 1024) != 0)  ri= ((ri & 0xF800) >> 1) | (ri & 0x3FF);
	if((numera &  512) != 0)  ri= ((ri & 0xFC00) >> 1) | (ri & 0x1FF);
	if((numera &  256) != 0)  ri= ((ri & 0xFE00) >> 1) | (ri & 0x0FF);
	if((numera &  128) != 0)  ri= ((ri & 0xFF00) >> 1) | (ri & 0x07F);
	if((numera &   64) != 0)  ri= ((ri & 0xFF80) >> 1) | (ri & 0x03F);
	if((numera &   32) != 0)  ri= ((ri & 0xFFC0) >> 1) | (ri & 0x01F);
	if((numera &   16) != 0)  ri= ((ri & 0xFFE0) >> 1) | (ri & 0x00F);
	if((numera &    8) != 0)  ri= ((ri & 0xFFF0) >> 1) | (ri & 0x007);
	if((numera &    4) != 0)  ri= ((ri & 0xFFF8) >> 1) | (ri & 0x003);
	if((numera &    2) != 0)  ri= ((ri & 0xFFFC) >> 1) | (ri & 0x001);
	if((numera &    1) != 0)  ri= ((ri & 0xFFFE) >> 1);

	if( GetBinPix(ri,(uint8_t *)SmaskF)!=0)
		SetBinPix(i,(uint8_t *)SmaskEx);
	}

}


//This procedure compute an optimal estimator for the selected bit plane
long SchCount3d2x3a(Raster2DAbstract & obr,Raster2DAbstract & PlaneX,uint8_t n,uint16_t & numera,TSmaskFull & SmaskF)
{
int x,y;
uint8_t mask;
int i;
unsigned long *a;		//array[0..4095] of longint;
long koutecku3f;
uint8_t *p0,*p1,*p2,*p3,*p4,*p5,*p6;
uint8_t *plane2a,*plane2b,*plane3,*plane4;
uint16_t lpl;

long qq[ProbeCells+1];
long lm,lg;
uint16_t c,num;
uint8_t qplanes;

uint8_t n2plane,n3plane,n4plane;

 if ((n>obr.GetPlanes()) || (obr.Data2D==NULL)) return(0);

 a=(unsigned long *)malloc(4096*sizeof(unsigned long));
 lpl=(obr.Size1D+7) / 8;

 plane2a=(uint8_t *)malloc(lpl);
 plane2b=(uint8_t *)malloc(lpl);
 plane3=(uint8_t *)malloc(lpl);
 plane4=(uint8_t *)malloc(lpl);

 memset(plane2a,0,lpl);
 memset(plane2b,0,lpl);
 memset(plane3,0,lpl);
 memset(plane4,0,lpl);

 Setn3n4forRGB(obr,n,n2plane,n3plane,n4plane);

 if(n2plane<=obr.GetPlanes()) obr.GetRowRaster(0)->Peel1Bit(plane2a,n2plane);

 memset(a,0,4096*sizeof(unsigned long));

 //FILE *f=fopen("c:\\temp\\aa.txt","wt"); //!!!!

 for(y=0;y<=obr.Size2D-2;y++)		//main row loop
   {
   if(n<obr.GetPlanes())
	{
	p3=plane2a;plane2a=plane2b;plane2b=p3;
	obr.GetRowRaster(y+1)->Peel1Bit(plane2a,n2plane);

	if(n+1<obr.GetPlanes())
	     {
	     obr.GetRowRaster(y+1)->Peel1Bit(plane3,n3plane);
	     if(n+2<obr.GetPlanes())
		{
		obr.GetRowRaster(y+1)->Peel1Bit(plane4,n4plane);
		}
	     }
	}

   p2=(uint8_t *)PlaneX.GetRowRaster(y+1)->Data1D; // data[y+1].b;
   p1=(uint8_t *)PlaneX.GetRowRaster(y)->Data1D;   //PlaneX.data[y].b;
   if(y==0) p0=p1;			//duplication of row 0
       else p0=(uint8_t *)PlaneX.GetRowRaster(y-1)->Data1D; //y or y-1 ????

   p3=plane2b;
   p4=plane2a;

   p5=plane3;
   p6=plane4;

   i=0;
   if((*p0 & 128) != 0) i+=128;
   if((*p1 & 128) != 0) i+=512;
   if((*p2 & 128) != 0) i+=2048;
   if((*p3 & 128) != 0) i+=8;
   if((*p4 & 128) != 0) i+=32;

   mask=64;
   for(x=0;x<=obr.Size1D-2;x++)
	   {
	   i= ((i >> 1) & 0x554);
	   if ((*p0 & mask)!=0) i |= 128;
	   if ((*p1 & mask)!=0) i |= 512;
	   if ((*p2 & mask)!=0) i |=2048;
	   if ((*p3 & mask)!=0) i |=   8;
	   if ((*p4 & mask)!=0) i |=  32;
	   if ((*p5 & mask)!=0) i |=   2;
	   if ((*p6 & mask)!=0) i |=   1;

	   //fprintf(f,"%d:%d %d p0:%d\n",x,y,i,*p0);

	   mask=mask >> 1;
	   if(mask==0)
		   {
		   mask=128;
		   p0++;
		   p1++;
		   p2++;
		   p3++;
		   p4++;
		   p5++;
		   p6++;
		   }
	   a[i]++;
	   }
   }


free(plane2a);
free(plane2b);
free(plane3);
free(plane4);

/*unsigned long jj=0;
for(i=0;i<=4095;i++)
  {fprintf(f,"%d: %ld\n",i,a[i]);jj+=a[i];}
fprintf(f,"%ld",jj);
fclose(f);*/

//-----------vypocty diskriminativnosti smeru---------

qplanes=11;
if(numera==0)
   {
RepeatReduction:
   num=1 << qplanes;

   koutecku3f=0;
   for(i=0;i<=num-1;i++)
	  if(a[i+num]>a[i]) koutecku3f+=a[i];
		       else koutecku3f+=a[i+num];

   memset(qq,0,sizeof(qq));
   for(x=0;x<=qplanes-1;x++)
     {
     c=1 << x;

     for(i=0;i<=num-1;i++)
	  {
	  lm=a[i]+a[i ^ c];
	  lg=a[i + num]+a[(i ^ c) + num];

	  if(lm<lg) qq[x]+=lm;
	       else qq[x]+=lg;
	  }
     }

     x=0;		//find the direction with less diskriminativness
     lm=0;
     for(i=qplanes-1;i>=0;i--)
	  {
	  qq[i]=(qq[i] >> 1) - koutecku3f;
	  if(lm>=qq[i])
		  {
		  lm=qq[i];
		  x=i;
		  }
	  }

		// bits of the table

     if((num > 2*qq[x])&&(qplanes>=1))
	  {				//remove direction i
	  c=1 << x;

	  y=0;
	  for(i=0;i<=(2 * num)-1;i++)
		  {
		  if((i & c) != 0) continue;
		  a[y]=a[i]+a[i | c];
		  y++;
		  }

	  for(i=0;i<=15;i++)
	       {
	       if(( (numera&(1<<i))!=0 )&&(x>=i)) x++;
	       }
	  numera |= 1 << x;
	  qplanes--;

	  goto RepeatReduction;
	  }
     }
     else {
	  for(x=qplanes-1;x>=0;x--)
		{
		num=1 << qplanes;
		if( (numera&(1<<(x-1)))!=0 )  //x in numera
			{
			c=1 << x;

			y=0;
			for(i=0;i<=(2 * num)-1;i++)
				{
				if((i & c) != 0) continue;
				a[y]=a[i]+a[i | c];
				y++;
				}
			qplanes--;
			}
		}
	  num=1 << qplanes;
	  }


memset(SmaskF,0,sizeof(SmaskF));
koutecku3f=0;
for(i=0;i<num;i++)
	{
	if(a[i+num]>a[i]) {
			  SetBinPix(i,(uint8_t *)SmaskF);
			  koutecku3f+=a[i];
			  }
		     else koutecku3f+=a[i+num];
	}

free(a);
return(koutecku3f);
}


// perform compression of the selected bit plane and known predictor
void SchComp3d2x3a(Raster2DAbstract & obr,Raster2DAbstract & PlaneC,uint8_t n,uint16_t numera,TSmaskFull & SmaskF)
{
int i,x,y;
uint8_t Rmask,OldRmask;
uint8_t *p0,*p1,*p2,*p3,*p4,*p5,*p6;
uint8_t *plane2a,*plane2b,*plane3a,*plane3b,*plane4a,*plane4b;
uint16_t lpl;
TSmaskFull SmaskEx;
uint8_t n2plane,n3plane,n4plane;

 if(n>obr.GetPlanes() || obr.Data2D==NULL) return;
 if(PlaneC.GetPlanes()!=1 || PlaneC.Data2D==NULL) PeelPlane(obr,PlaneC,n-1);

 ExplodeBitFlags(numera,SmaskF,SmaskEx);
 lpl=(obr.Size1D+7) / 8;

 plane2a=(uint8_t *)malloc(lpl);
 plane2b=(uint8_t *)malloc(lpl);
 plane3a=(uint8_t *)malloc(lpl);
 plane3b=(uint8_t *)malloc(lpl);
 plane4a=(uint8_t *)malloc(lpl);
 plane4b=(uint8_t *)malloc(lpl);

 memset(plane2a,0,lpl);
 memset(plane2b,0,lpl);
 memset(plane3a,0,lpl);
 memset(plane3b,0,lpl);
 memset(plane4a,0,lpl);
 memset(plane4b,0,lpl);

 Setn3n4forRGB(obr,n,n2plane,n3plane,n4plane);

 obr.GetRowRaster(obr.Size2D-1)->Peel1Bit(plane2b,n2plane);
 obr.GetRowRaster(obr.Size2D-1)->Peel1Bit(plane3b,n3plane);
 obr.GetRowRaster(obr.Size2D-1)->Peel1Bit(plane4b,n4plane);

 for(y=obr.Size2D-2;y>=0;y--)		//main rows loop
	{
	if(n<obr.GetPlanes())
		{
		p3=plane2a;plane2a=plane2b;plane2b=p3;
		obr.GetRowRaster(y)->Peel1Bit(plane2b,n2plane);

		if(n+1<obr.GetPlanes())
		     {
		     p5=plane3a;plane3a=plane3b;plane3b=p5;
		     obr.GetRowRaster(y)->Peel1Bit(plane3b,n3plane);
		     if(n+2<obr.GetPlanes())
			{
			p6=plane4a;plane4a=plane4b;plane4b=p6;
			obr.GetRowRaster(y)->Peel1Bit(plane4b,n4plane);
			}
		     }
		}

	p2=(uint8_t *)PlaneC.GetRowRaster(y+1)->Data1D;
	p1=(uint8_t *)PlaneC.GetRowRaster(y)->Data1D;
	if(y==0) p0=p1;
	    else p0=(uint8_t *)PlaneC.GetRowRaster(y-1)->Data1D;


	p3=plane2b;
	p4=plane2a;

	p5=plane3a;
	p6=plane4a;

	p0+=(obr.Size1D-1) >> 3;
	p1+=(obr.Size1D-1) >> 3;
	p2+=(obr.Size1D-1) >> 3;
	p3+=(obr.Size1D-1) >> 3;
	p4+=(obr.Size1D-1) >> 3;
	p5+=(obr.Size1D-1) >> 3;
	p6+=(obr.Size1D-1) >> 3;

	Rmask=128 >> ((obr.Size1D-1) & 7);

	i=0;
	if( (*p0 & Rmask) != 0) i+=64;
	if( (*p1 & Rmask) != 0) i+=256;	//obr.Size1D-1 column
	if( (*p2 & Rmask) != 0) i+=1024;
	if( (*p3 & Rmask) != 0) i+=4;
	if( (*p4 & Rmask) != 0) i+=16;

	OldRmask=Rmask;
	Rmask=Rmask << 1;
	if(Rmask==0) {
		     Rmask=1;
		     p0--;
		     p1--;
		     p2--;
		     p3--;
		     p4--;
		     }

	for(x=obr.Size1D-2;x>=0;x--)
		{
		i= ((i << 1) & 0xAA8);
		if((*p0 & Rmask)!=0)  i |=  64;
		if((*p1 & Rmask)!=0)  i |= 256;
		if((*p2 & Rmask)!=0)  i |=1024;
		if((*p3 & Rmask)!=0)  i |=   4;
		if((*p4 & Rmask)!=0)  i |=  16;

		if((*p5 & OldRmask)!=0)	i |= 2;
		if((*p6 & OldRmask)!=0) i |= 1;

		OldRmask=Rmask;
		if(OldRmask==1)
		       {
		       p5--;
		       p6--;
		       }
		Rmask=Rmask << 1;
		if(Rmask==0)
			{
			Rmask=1;
			p0--;
			p1--;
			p2--;
			p3--;
			p4--;
			}


		if ((GetBinPix(i & 2047,(uint8_t *)SmaskEx)!=0))
			{
			if (i>=2048)  PlaneC.SetValue2D(x+1,y+1,0); //px=1
				 else PlaneC.SetValue2D(x+1,y+1,1); //px=0
			}

		}
	}

free(plane2a);
free(plane2b);
free(plane3a);
free(plane3b);
free(plane4a);
free(plane4b);
}



//RLC Compression of the 1st row and 1st column
unsigned Schless1d(Raster2DAbstract & obr,int x,int y,unsigned kx,unsigned ky,uint8_t & S1)
{
uint8_t M;
int i,Max;
unsigned a[4];
unsigned koutecku;

 if(obr.Data2D==NULL || obr.GetPlanes()!=1) return(0);

 memset(a,0,sizeof(a)); // pozor !!!

 Max = 0;
 if(kx==0) Max=obr.Size2D-2;
 if(ky==0) Max=obr.Size1D-2;

 for(i=0;i<=Max;i++)
	{
	M=   obr.GetValue2D(x+(kx & i),y+(ky & i))+
	   2*obr.GetValue2D(x+(kx & (i+1)),y+(ky & (i+1)));
	a[M]++;
	}

 S1=0;
 koutecku=0;

 for(i=0;i<=1;i++)
	{
	if(a[i+2]>a[i])
		{
		S1=S1 | (1 << i);
		koutecku+=a[i];
		}
		else koutecku+=a[i+2];
	}


 for(i=Max;i>=0;i--)
	{
	M=obr.GetValue2D(x+(kx & i),y+(ky & i));

	if( ( (S1 & (1 << M))!=0 ) ^ (obr.GetValue2D(x+(kx & (i+1)),y+(ky & (i+1)))==1))
		     obr.SetValue2D(x+(kx & (i+1)),y+(ky & (i+1)),1);
		else obr.SetValue2D(x+(kx & (i+1)),y+(ky & (i+1)),0);
	}

return(koutecku);
}


//RLC Decompression of the 1st row and 1st column
void DeSchless1d(Raster2DAbstract & obr,uint16_t x,uint16_t y,uint16_t kx,uint16_t ky,uint8_t S1)
{
uint8_t M;
uint16_t Max;
uint16_t i;

 Max = 0;
 if(kx==0) Max=obr.Size2D-2;
 if(ky==0) Max=obr.Size1D-2;

 for(i=0;i<=Max;i++)
	{
	M=obr.GetValue2D(x+(kx & i),y+(ky & i));

	if ( (((1<<M) & S1)!=0) ^ (obr.GetValue2D(x+(kx & (i+1)),y+(ky & (i+1)))==1))
		     obr.SetValue2D(x+(kx & (i+1)),y+(ky & (i+1)),1);
		else obr.SetValue2D(x+(kx & (i+1)),y+(ky & (i+1)),0);
	}
}


//This procedure performs a decompression of the selected compressed bit plane and known predictor
void DeSchless3d2x3a(Raster2DAbstract & obr,Raster2DAbstract & PlaneC,uint8_t n,uint16_t & numera,TSmaskFull & SmaskF)
{
uint16_t x,y;
uint8_t mask;
uint16_t i;
uint8_t *p0,*p1,*p2,*p3,*p4,*p5,*p6;
void *plane2a,*plane2b,*plane3,*plane4;
uint16_t lpl;
TSmaskFull SmaskEx;
uint8_t n2plane,n3plane,n4plane;

 if(n>obr.GetPlanes() || obr.Data2D==NULL || PlaneC.Data2D==NULL) return;
 ExplodeBitFlags(numera,SmaskF,SmaskEx);

 lpl=(obr.Size1D+7) / 8;
 plane2a=malloc(lpl);
 plane2b=malloc(lpl);
 plane3= malloc(lpl);
 plane4= malloc(lpl);

 memset(plane2a,0,lpl);
 memset(plane2b,0,lpl);
 memset(plane3,0,lpl);
 memset(plane4,0,lpl);

 Setn3n4forRGB(obr,n,n2plane,n3plane,n4plane);

 obr.GetRowRaster(0)->Peel1Bit(plane2a,n2plane);

 for(y=0;y<=obr.Size2D-2;y++)	//main loop through rows
	{
	if(n<obr.GetPlanes())
		{
		if((numera & (4|8|16|32)) != (4|8|16|32))
		  {
		  p3=(uint8_t *)plane2a;plane2a=plane2b;plane2b=p3;
		  obr.GetRowRaster(y+1)->Peel1Bit(plane2a,n2plane);
		  }

		if(n+1<obr.GetPlanes())
		     {
		     if((numera & 2)!=2)
			{
			obr.GetRowRaster(y+1)->Peel1Bit(plane3,n3plane);
			}

		     if(n+2<obr.GetPlanes())
			{
			if((numera & 1)!=1)
			    {
			    obr.GetRowRaster(y+1)->Peel1Bit(plane4,n4plane);
			    }
			}
		     }
		}


	p2=(uint8_t *)PlaneC.GetRowRaster(y+1)->Data1D;
	p1=(uint8_t *)PlaneC.GetRowRaster(y)->Data1D;
	if(y==0) p0=p1;
	    else p0=(uint8_t *)PlaneC.GetRowRaster(y-1)->Data1D;

	p3=(uint8_t *)plane2b;
	p4=(uint8_t *)plane2a;

	p5=(uint8_t *)plane3;
	p6=(uint8_t *)plane4;


	i=0;
	if( (*p0 & 128) != 0 ) i+=128;
	if( (*p1 & 128) != 0 ) i+=512;
	if( (*p2 & 128) != 0 ) i+=2048;
	if( (*p3 & 128) != 0 ) i+=8;
	if( (*p4 & 128) != 0 ) i+=32;

	mask=64;

	for(x=0;x<=obr.Size1D-2;x++)
		{
		i= ((i >> 1) & 0x554);
		if( (*p0 & mask)!=0 ) i=i |  128;
		if( (*p1 & mask)!=0 ) i=i |  512;
		if( (*p2 & mask)!=0 ) i=i | 2048;
		if( (*p3 & mask)!=0 ) i=i |    8;
		if( (*p4 & mask)!=0 ) i=i |   32;
		if( (*p5 & mask)!=0 ) i=i |    2;
		if( (*p6 & mask)!=0 ) i=i |    1;

		if ((GetBinPix(i  &  2047,SmaskEx)!=0))
			{
			i=i ^ 2048;
			*p2=*p2 ^ mask;
			}

		mask=mask >> 1;
		if(mask==0)
			{
			mask=128;
			p0++;
			p1++;
			p2++;
			p3++;
			p4++;
			p5++;
			p6++;
			}

		}
	}

free(plane2a);
free(plane2b);
free(plane3);
free(plane4);
}


/// This procedure writes number to the bitstream
/// 1 ... 00b
/// 2 ... 01b
/// 1_0_0	3 ... 1_0_1
/// 4 ... 11_0_00	5...11_0_01	6...11_0_10	7...11_0_11
void WriteNumber(bits_t *pbits, long x)
{
unsigned i;
long a;

if(x<=0) return;
x--;
if(x<2) {
	x=x << 1;
	sq_wrn(pbits,(uint16_t)x,2);
	return;
	}
i=0;
a=x;
while(a>0)
	{
	a>>=1;
	i++;
	}

a=0xFFFFFFFFl;
i--;
if(i<17) {
	 sq_wrn(pbits,(uint16_t)a,i);
	 sq_wrn(pbits,0,1);
	 sq_wrn(pbits,(uint16_t)x,i);		// Store n-1 bits, nth bit is always 1.
	 }
    else {
	 sq_wrUnl(pbits,(uint8_t *)&a,i);
	 sq_wrn(pbits,0,1);
	 //sq_wrUnl(pbits,(uint8_t *)&x,i);	// this does not work on HI endian.
         sq_wrn(pbits,(uint16_t)(x&0xFFFF),16);  // Store n-1 bits, nth bit is always 1.
         sq_wrn(pbits,(uint16_t)(x>>16),i-16);
	 }
return;
}


//this procedure is used for compression of the huffman description table
//prefixes - used prefixes inside number stream
unsigned PackSizes(bits_t *Bit, uint8_t *b, uint16_t MaxItems)
{
int i,j;
short x,Max,MaxDif,t,lastS;
unsigned FinalSize;

FinalSize=0;

lastS=2;
Max=b[0];
MaxDif=0;

for(i=1;i<=MaxItems;i++)
	{
	t=b[i];
	if(t!=0) {
		 if((abs(lastS-t))>MaxDif) MaxDif=abs(lastS-t);
		 lastS=t;
		 }
	if(t>Max) Max=t;
        }

t=127;		//zatim neumim
if(Max<=30) t= 2;  // primy kod 0..30 RLC 31
if(Max<=14) t= 1;  // primy kod 0..14 RLC 15
if(Max<= 6) t= 0;  // primy kod 0.. 6 RLC  7

if((MaxDif<= 6)&&(Max>6)) t=5;
if (MaxDif<= 2)		  t=4;

if(t==127) return(0);		//I cannot store


if(Bit!=NULL) sq_wrn(Bit,t,3);
FinalSize+=3;


i=0;
Max=(t & 3) + 3;
if(t>=4)
     {			// Store histogram diff accepted
     lastS=2;

     if(Bit!=NULL)  WriteNumber(Bit,b[0]+1);
     FinalSize+=SizeNum(b[0]+1);

     for(i=1;i<=MaxItems;i++)
	{
	x=b[i];
	if(x==0) b[i]=ZeroValues[Max];
	    else {
		 b[i]=x-lastS+Aditors[Max];
		 lastS=x;
		 }
	}

      i=1;
      }


lastS=2;
while(i<=MaxItems)	// Store histogram RLC; no diff in this place
	{
	x=b[i];

	j=i;
	if(x==lastS)
		{
		while ((j<=MaxItems)&&(b[j]==lastS))
			{
			j++;
			}
		}


	if(j-i>1 ) 		//komprese se vyplati pro opakovani >= 3x
		 {
		 if(Bit!=NULL)
			{
			sq_wrn(Bit,Repeaters[Max],Max);
			WriteNumber(Bit,j-i-1);
			}
		 FinalSize+=Max;
		 FinalSize+=SizeNum(j-i-1);
		 i=j-1;
		 }
		else
		{
		if(Bit!=NULL)  sq_wrn(Bit,x,Max);
		FinalSize+=Max;
		}

	lastS=x;
	i++;
	}


if(t>=4)		// inverse diff process
     {
     lastS=2;
     for(i=1;i<=MaxItems;i++)
	      {
	      if(b[i]==ZeroValues[Max]) b[i]=0;
	      else
		{
		b[i]-=Aditors[Max];
		b[i]+=lastS;
		lastS=b[i];
		}
	      }
     }

return(FinalSize);
}


/// This procedure reads a number from the bitstream.
long ReadNumber(bits_t *pbits)
{
static const uint8_t exchange[4] = {0,2,1,3};
uint16_t i;
long n;

i = sq_rdn(pbits,2);
i = exchange[i];
if(i<2)	return(i+1);			// i=0,1 ---> 1,2

i--;
if(i==2)
{
  while(sq_rdn(pbits,1)!=0) i++;
}

if(i>16)
    n = sq_rdn(pbits,16) | (sq_rdn(pbits,i-16)>>16);
else
    n = sq_rdn(pbits,i);
return(((long)(1l) << i) + n + 1l);
}


/// This procedure is used for decompression of the huffman description table.
static void UnPackSizes(bits_t *Bit, uint8_t *b, unsigned MaxItems, short t)
{
int i,j;
short Vtyp,typ,lastS;

typ = sq_rdn(Bit,3);
i=0;
j=0;

if(typ>=4)
	{
	b[0]=ReadNumber(Bit)-1;
	i=1;
	}
lastS=2;

Vtyp=(typ & 3) + 3;
while(i<=MaxItems)
	 {
//	 b[i].Number=i;

	 if(j>0)
	       {
	       j--;
	       b[i]=lastS;
	       i++;
	       continue;
	       }

	 b[i]=sq_rdn(Bit,Vtyp);
	 if(b[i]==Repeaters[Vtyp])
		   {
		   b[i]=lastS;
		   j=ReadNumber(Bit);
		   i++;
		   continue;
		   }
	 lastS=b[i];
	 i++;
	 }


if(typ>=4)
    {
    lastS=2;
    for(i=1;i<=MaxItems;i++)
	      {
	      if(b[i]==ZeroValues[Vtyp]) b[i]=0;
		       else
		       {
		       b[i]-=Aditors[Vtyp];
		       b[i]=lastS+b[i];
		       lastS=b[i];
		       }
	      }
    }
}


#if SupportFTG>=4 || SupportFTG==2		/////////  READ //////////


/** This procedure evaluates huffman codes. */
static int HufCount(huf_wr_t *b, unsigned HItems)
{
unsigned i;
long x,y;
unsigned BitHistogram[17];

 memset(BitHistogram,0,sizeof(BitHistogram));

 for(i=0;i<HItems;i++)
   BitHistogram[b->ln[i]]++;

 y=0;
 BitHistogram[0]=0;
 for(i=1;i<=16;i++)
   {
   x = BitHistogram[i];

   if(x!=0) BitHistogram[i] = y;
   y <<= 1;
   y+=x;
   }

	/* Fill calculated Huffman codes */
/*
 for(i=0; i<HItems; i++)
   {
   if(b->ln[i]==0) b->cod[i] = 0;
	      else b->cod[i] = swap_bits_order(BitHistogram[b->ln[i]],b->ln[i]);
   BitHistogram[b->ln[i]]++;
   }
*/

 return y==65536l;
}


/// This procedure reads hufman encoded number from the bitstream
static long ReadHufNum(bits_t *pbits, const huf_rd_t *phuf,uint8_t HprPos)
{
uint16_t i;
long n;

 i=0;

 n = sq_rdh1(pbits,phuf);
 if(n>0) return(n);


 i=HprPos;
 while(sq_rdn(pbits,1)!=0) i++;
 if(i<=16) n=sq_rdn(pbits,i);
      else {
	   n=sq_rdn(pbits,16);
	   n+= 65536l * sq_rdn(pbits,i-16);
	   }

 return(((long)1 << i) + n + 1);
}


Image LoadPictureFTG(const char *Name)
{
TImageStruct NewImage;
TObjectReader *ActualReader;
RasterData RDataDesc;
PaletteData PaletteDesc;
int i,x,y;
int ldblk;
bits_t BitStream;
huf_rd_t *phuf;
uint8_t *CompBuffer=NULL;
uint16_t renums;
TSmaskFull mFull;
uint8_t Smask1dx,Smask1dy;
uint8_t isHuff,HufN,HufTblSize;
uint8_t *bha=NULL;
long NewX;
Raster2DAbstract *RasRead=NULL;
APalette *Palette=NULL;
PropertyItem *Prop;
Image Img;
Image *CurrImg = &Img;
PropertyList PropList;

 if((i=NewImage.AttachFile(Name))!=0) return(Img);
 ActualReader=&NewImage.RootReader;

//printf("File opened\n\r");

 while(ActualReader->ReadObject(ActualReader->NextObjPos)>=0)
   {
//	printf("Object type %d, size %d\n\r",ActualReader->ActualObj.type,ActualReader->ActualObj.size);
   switch(ActualReader->ActualObj.type)
   {
     case 1:if(ActualReader->ActualObj.size > 0)
            {
              Prop = new PropertyItem("comment",ActualReader->ActualObj.size);
              goto HandleProp;
            }
	    break;
     case 4:{
            if(LoadFtgData(NewImage.ImgFile,RDataDesc)!=7) break;
	    // printf("  Raster Image:: Size1D:%d  sizeY:%d  planes:%d  compression:%d \n\r",RDataDesc.Size1D,RDataDesc.SizeY,RDataDesc.GetPlanes(),RDataDesc.compression);

	    if(RasRead!=NULL)
            {
	      if(!CurrImg->isEmpty())
              {
	        CurrImg->Next = new Image;
                CurrImg = CurrImg->Next;
	      }
              CurrImg->AttachRaster(RasRead); RasRead=NULL;
              CurrImg->AttachPalette(Palette);
              CurrImg->AttachPropList(PropList);
            }

            if(RDataDesc.interpretation==Inside_RGB && RDataDesc.planes==32)
              RasRead = CreateRaster2DRGBA(RDataDesc.SizeX,RDataDesc.SizeY,8);
	    if(RDataDesc.interpretation==Inside_RGB && RDataDesc.planes==24)
              RasRead = CreateRaster2DRGB(RDataDesc.SizeX,RDataDesc.SizeY,8);
            else
	      RasRead = CreateRaster2D(RDataDesc.SizeX,RDataDesc.SizeY,RDataDesc.planes);
            if(RasRead==NULL) break;
	    if(RasRead->Data2D==NULL) break;
	    //	    printf("         OK to load & Init memory struct\n\r");

	    if(RDataDesc.compression==0)
	    {
	      ldblk = (int)(((long)RDataDesc.SizeX*RDataDesc.planes+7)/8);
	      for(y=0; y<RDataDesc.SizeY; y++)
	      {
		if(fread(RasRead->GetRow(y),1,ldblk,NewImage.ImgFile)!=ldblk)
		{
		  if(RasRead) {delete RasRead; RasRead=NULL;}
	          goto AbortReading;
		}
	      }
	   } else
	     {
             Raster2DAbstract *PlaneX = NULL;;
	     ldblk = (RasRead->GetSize1D()+7) / 8;

	     if(bha==NULL) bha = (uint8_t *)malloc(sizeof(bHuf));
	     phuf = (huf_rd_t *)malloc(sizeof(huf_rd_t));
	     CompBuffer = (uint8_t *)malloc(CBufSize);

	     if(CompBuffer==NULL || bha==NULL || phuf==NULL)
	     {
	       if(RasRead) {delete RasRead; RasRead=NULL;}
	       goto AbortReading;
	     }
								/* - sizeof struct */
	     sq_rdinit(&BitStream,CompBuffer,CBufSize,ActualReader->ActualObj.size,NewImage.ImgFile);

	     for(i=RasRead->GetPlanes();i>=1;i--)
		 {
                 if(PlaneX==NULL)
		     PlaneX = CreateRaster2D(RasRead->GetSize1D(),RasRead->Size2D,1);
		 PlaneX->Cleanup();	//fill zeros to all data

		 if(sq_rdn(&BitStream,1))
		   {			//plane is compressed
		   if((RDataDesc.compression & SchMask) < Schless_3DVar)
		     {
		     if(sq_rdn(&BitStream,1)) renums=1 | 2 | 64 | 128;
		                         else renums=1 | 2 | 4 | 8 | 16 | 32 | 64 | 128;
		     }
		   else renums = sq_rdn(&BitStream,16);

		   y=0;
		   for(x=1; x<=(1<<ProbeCells); x<<=1) 
		     if(!(x & renums)) y++;

		   memset(mFull,0,sizeof(mFull));
		   sq_rdUnl(&BitStream,mFull,1 << y);

		   Smask1dx=sq_rdn(&BitStream,2);
		   Smask1dy=sq_rdn(&BitStream,2);

		   if((isHuff=sq_rdn(&BitStream,1))==1)
		     {
		     HufN = sq_rdn(&BitStream,3);
		     HufTblSize = 0;

		     UnPackSizes(&BitStream,bha,Hu2Sizes[HufN],HufTblSize);

		     x = Hu2Sizes[HufN]+1;
		     bha[x]=0xFF;

		   if(!sq_rdhufi(phuf,x,8,bha))
		   {
                     printf("sq_rdhufi failed\n");
		     goto ExitWriteRas;
		   }

		   HufN += Hu2Prefix[0]; 	//prefix length in bites
		 }

		 NewX=-1;
		 y=0;
		 while(y<PlaneX->Size2D)
		   {
		   if(isHuff==1) NewX+=ReadHufNum(&BitStream,phuf,HufN);
			    else NewX+=ReadNumber(&BitStream);

		   if(NewX>=PlaneX->GetSize1D()-1)
		     {
		     while(NewX>=PlaneX->GetSize1D())		//end of row
		       {
		       y++;
		       NewX-=PlaneX->GetSize1D();
		       }

		   if(NewX==PlaneX->GetSize1D()-1 && y==PlaneX->Size2D-1)
		     {
		     PlaneX->SetValue2D((int)NewX,y,1);   //last pixel
		     break;
		     }
		   }
		 if(y<=PlaneX->Size2D-1) PlaneX->SetValue2D((int)NewX,y,1);
		 }

		 if((NewX!=0)&&(NewX!=PlaneX->GetSize1D()-1))
		 {
		   printf("Error NewX:%ld!=%u\n",NewX,PlaneX->GetSize1D()-1);
		   goto ExitWriteRas;
		 }

//!!!!!!!!
//		{char *Name=strdup("o:\\temp\\0_.bmp"); Name[8]='0'+i;
//		PlaneX->UsageCount=1;
//		SavePictureBMP(Name,PlaneX); 
//		free(Name);}	///!!!!!!!!!!!!!!!!!!!!!!

		 DeSchless1d(*PlaneX,0,0,0xFFFF,0,Smask1dx);
		 DeSchless1d(*PlaneX,0,0,0,0xFFFF,Smask1dy);

		 DeSchless3d2x3a(*RasRead,*PlaneX,i,renums,mFull);
	       }
	       else for(y=0;y<RasRead->Size2D;y++)	//plane is stored uncompressed
	       {
	         sq_rdUnl(&BitStream,(uint8_t *)PlaneX->GetRowRaster(y)->Data1D,RasRead->GetSize1D());
	         x=(PlaneX->GetSize1D()-1)>>3;
	         ((uint8_t *)(PlaneX->GetRowRaster(y)->Data1D))[x] = swap_bits_xlat[((uint8_t *)(PlaneX->GetRowRaster(y)->Data1D))[x]];
	       }

	       JoinPlane(*PlaneX,*RasRead,i-1);
	     }

ExitWriteRas:
	     free(CompBuffer);CompBuffer=NULL;
	     free(bha); bha=NULL;
	     free(phuf); phuf=NULL;
             if(PlaneX) {delete PlaneX;PlaneX=NULL;}	     
	     }

           fseek(NewImage.ImgFile,ActualReader->NextObjPos,SEEK_SET);
	   break;
	   }

     case 20:
           {
	   if(fread(&PaletteDesc,sizeof(PaletteData),1,NewImage.ImgFile)!=1) break;

//	   printf("  Palette:: items:%d  typ:%d  planes:%d\n\r",PaletteDesc.items,PaletteDesc.typ,PaletteDesc.destPlanes);

	   if(PaletteDesc.items!=0)
	   {
             switch(PaletteDesc.typ)
             {
	       case 3: Palette=BuildPalette(PaletteDesc.items,8); break;
               case 6: Palette=BuildPalette(PaletteDesc.items,16); break;
             }
	     if(!Palette) break;

	     for(i=0;i<Palette->Size1D;i++)
	     {
		Palette->setR(i,fgetc(NewImage.ImgFile));
		Palette->setG(i,fgetc(NewImage.ImgFile));
		Palette->setB(i,fgetc(NewImage.ImgFile));
	     }
	   }
	   break;
	   }

     case 40: Prop = new PropertyItem(ExifPreffix,ActualReader->ActualObj.size);
              goto HandleProp;
     case 41: Prop = new PropertyItem("XMP",ActualReader->ActualObj.size);
	      goto HandleProp;
     case 42: Prop = new PropertyItem("ICCP",ActualReader->ActualObj.size);
HandleProp:   if(Prop->Data==NULL)
              {
                delete Prop; Prop=NULL;
                break;
              }
	      if(fread(Prop->Data,1,ActualReader->ActualObj.size,NewImage.ImgFile) != ActualReader->ActualObj.size)
	      {
		delete Prop; Prop=NULL;
                break;
	      }
	      PropList.AddProp(Prop); Prop=NULL;
              break;
     }
   }

AbortReading:
 if(bha!=NULL) free(bha);
 if(!CurrImg->isEmpty())
 {
   CurrImg->Next = new Image;
   CurrImg = CurrImg->Next;
 }
 CurrImg->AttachRaster(RasRead);
 CurrImg->AttachPalette(Palette);
 CurrImg->AttachPropList(PropList);
return(Img);
}
#endif


#if SupportFTG>=3		/////////  WRITE //////////


/** This procedure writes huffmam encoded number to the bitstream. */
static void WriteHufNum(bits_t *pbits, const huf_wr_t *phuf, uint32_t x, uint8_t HufIdx)
{
unsigned i;
uint32_t a;

if(x==0) return;
if(x<=Hu2Sizes[HufIdx])
	{
	sq_wrh(pbits,phuf,(int)x);
	return;
	}

sq_wrh(pbits,phuf,0);
x--;
i=0;
a=x;
while(a>0)
	{
	a>>= 1;
	i++;
	}

a=0xFFFFFFFFl;
sq_wrn(pbits,a,i-1-Hu2Prefix[HufIdx]);
sq_wrn(pbits,0,1);
sq_wrn(pbits,(int)x,i-1);
}


int SavePictureFTG(const char *Name, const Image & Img)
{
ImgObject ObjLeader;
RasterData RDataDesc;
PaletteData PaletteDesc;
unsigned ldblk;
bits_t BitStream;
uint16_t renums;
long koutecku,lastX;
uint8_t Smask1d;
TSmaskFull mFull;
uint8_t isHuffman,HufN;
uint8_t Rotm,*pb;
count_t *Histogram=NULL;
count_t HufZero,HufMax;
huf_wr_t *HufStruct=NULL;
ch_tab_t *ch_tab=NULL;
int MaxHuTbSize;
unsigned BcountLen[MAX_BITS];
unsigned StartPos;
const PropertyItem *Prop;
FILE *f;
int SavePictureFTG_;
int i,x,y;
Raster2DAbstract *RasWr;
Raster2DAbstract *PlaneX=NULL;
const Image *CurrImg;

 CurrImg = &Img;
 while(CurrImg->isEmpty())
 {
   CurrImg = CurrImg->Next;
   if(CurrImg==NULL) return(ErrEmptyRaster);
 } 

 SavePictureFTG_=0;
 if((f=fopen(Name,"wb"))==NULL) return(ErrOpenFile);

 if(fwrite(FTG_IDENTIF,1,4,f)!=4) return(ErrWriteFile);	//write image Header to disk
 if(WrWORD_LoEnd(2,f)!=2) return(ErrWriteFile);

 while(CurrImg != NULL)
 {
   RasWr = CurrImg->Raster;
   if(RasWr==NULL || RasWr->Data2D==NULL)
   {
     CurrImg = CurrImg->Next;
     continue;
   }

   memset(&RDataDesc,0,sizeof(RDataDesc));
   RDataDesc.compression = FtGrCompression;
   if((RasWr->GetSize1D()<8)&&(RasWr->Size2D<8)) RDataDesc.compression=0;

   BcountLen[0] = 0;
   Prop = Img.Properties.Find("COMPRESSION",BcountLen[0]);	// Reference only the first frame for arguments.
   if(Prop && Prop->DataSize>=sizeof(int))
   {
     i = *(int*)Prop->Data;
     if(i==0) RDataDesc.compression=0;
   }

   ldblk = (RasWr->GetPlanes()*RasWr->GetSize1D()+7) / 8;

   StartPos = ftell(f);
   ObjLeader.type = 4;		//=image;
   if(RDataDesc.compression == 0)
     ObjLeader.size = ldblk*RasWr->Size2D + SIZE_RAS_DATA;	// The size is known
   else
     ObjLeader.size = ~1;
   if(SaveFtgObject(f,ObjLeader)!=SizeImgObject) return(-12); //error - cannot write object descriptor

   RDataDesc.SizeX = RasWr->GetSize1D();	//Rows
   RDataDesc.SizeY = RasWr->Size2D;		//Cols
   RDataDesc.planes = RasWr->GetPlanes();
   if(RasWr->Channels() >= 3) RDataDesc.interpretation=Inside_RGB;
   else if(CurrImg->Palette!=NULL && CurrImg->Palette->Size1D>0) RDataDesc.interpretation=Inside_Palette;
 
   SaveFtgData(f,RDataDesc);

/*  !!!!!!!!!!!!!!! DODELAT !!!!!!!!!!!!!!!!
   if(ImgRec.Compression!=0)
   {
     if(ImgRec.Compression & ReMapPal)!=0)
     {
       if(p.palette!=NULL) ReMapPalette(p)
       else
	else if (ImgRec.Compression & GrayCode)!=0 then Bin2Gray(p);
	 end;
   } */

   if(RDataDesc.compression==0)
   {
     for(y=0; y<RasWr->Size2D; y++)			//no compression mode
     {
       fwrite(RasWr->GetRowRaster(y)->Data1D,ldblk,1,f);
//	AlineProc(y,p);
     }
   }
   else 
   {
     uint8_t *CompBuffer = (uint8_t *)malloc(CBufSize);
     if(CompBuffer==NULL) return -13;
     sq_wrinit(&BitStream,CompBuffer,CBufSize,0xFFFFFFFl,f);

     PlaneX = CreateRaster2D(RasWr->GetSize1D(),RasWr->Size2D,1);
     if(PlaneX==NULL) {free(CompBuffer);return(-13);}

     for(i=RasWr->GetPlanes();i>=1;i--)
     {
       PeelPlane(*RasWr,*PlaneX,i-1);
       if(PlaneX->Data2D==NULL) return(-13);

       renums = 0;
       if((RDataDesc.compression & SchMask) < Schless_3D) renums=1 | 2 | 4 | 8 | 16 | 32 | 64 | 128;
       else {
	  if((RDataDesc.compression & SchMask) < Schless_3DVar)
	  renums=1 | 2 | 64 | 128;
       }

       koutecku = SchCount3d2x3a(*RasWr,*PlaneX,i,renums,mFull)+1;

       y = 0;
       for(x=1; x<=(1 << ProbeCells); x<<=1)
         if(!(x & renums)) y++;

       if( ((2.5*koutecku)+(1 << y)) < ((long)(PlaneX->GetSize1D()-1))*(PlaneX->Size2D-1))
       {		//is it a good think to compress this plane?
         sq_wrn(&BitStream,1,1);

         if((RDataDesc.compression & SchMask) < Schless_3DVar)
         {
	   if(renums==(1 | 2 | 4 | 8 | 16 | 32 | 64 | 128))
		sq_wrn(&BitStream,0,1);
	   else sq_wrn(&BitStream,1,1);
         }
         else sq_wrn(&BitStream,renums,16);

         sq_wrUnl(&BitStream,mFull,1 << y);

         SchComp3d2x3a(*RasWr,*PlaneX,i,renums,mFull);

         Schless1d(*PlaneX,0,0,0xFFFF,0,Smask1d); 	//0-th row
         sq_wrn(&BitStream,Smask1d,2);
         Schless1d(*PlaneX,0,0,0,0xFFFF,Smask1d);	//0-th column
         sq_wrn(&BitStream,Smask1d,2);

         isHuffman = 0;
//
//		char *Name="c:\\temp\\0_.bmp"; Name[8]='0'+i;
//		PlaneX->UsageCount=1;
//		SavePictureBMP(Name,PlaneX); ///!!!
//	 -------Huffmann compression----}
	 if((RDataDesc.compression & HufMask)!=0)
			{
			if(Histogram==NULL) Histogram=(count_t *)malloc(MAX_CODES*sizeof(count_t));
			if(HufStruct==NULL) HufStruct=(huf_wr_t *)malloc(sizeof(huf_wr_t));
			if(ch_tab==NULL)    ch_tab=(ch_tab_t *)malloc(MAX_CODES*sizeof(ch_tab));
			if((Histogram==NULL)||(HufStruct==NULL)||(ch_tab==NULL)) goto NoHuff;

			memset(Histogram,0,MAX_CODES*sizeof(count_t));
			lastX=-1;
			for(y=0; y<PlaneX->Size2D; y++)
			{
			  Rotm = 128;
			  pb = (uint8_t *)PlaneX->GetRowRaster(y)->Data1D;

			  for(x=0; x<PlaneX->GetSize1D(); x++)
			  {
			    if((*pb & Rotm) != 0)
			    {
				lastX=(long)(x)-lastX;
				if(lastX>=MAX_CODES) lastX=0;
				Histogram[(int)lastX]++;
				lastX=x;
			    }

 			    Rotm=Rotm >> 1;
			    if(Rotm==0)
			    {
			      Rotm=128;
			      pb++;
			    }
			  }
			  lastX = lastX - PlaneX->GetSize1D();
			}
			if(lastX<0)
			{
			  lastX= -lastX;
			  if(lastX>=MAX_CODES) lastX=0;
			  Histogram[(int)lastX]++;
			}

			HufZero = Histogram[0];
			HufMax = 0;
			y = 0;

			if((RDataDesc.compression & HufMask) >= HuffVar)
			  {
			    /*Iterate through all supported splitting a number se to 2 groups */
			    /* first group is encoded by huffman and a second group is stored in logaritmical code */
			  for(HufN=0;HufN<=6;HufN++)
			    {
			    Histogram[0]=HufZero;
			    for(x=Hu2Sizes[HufN]+1;x<MAX_CODES;x++) Histogram[0]+=Histogram[x];

                            memset(HufStruct->ln,0,MAX_CODES);	// HufStruct->ln contains uninitialised values.
			    x = sq_huffman(Histogram,HufStruct->ln,BcountLen,Hu2Sizes[HufN]+1,ch_tab);

				/* Calculate data stream difference-size in bits. */
			    lastX = Histogram[0] * (Hu2Prefix[HufN] - HufStruct->ln[0]);

			    // 1st set is Huffman encoded and its elements ere Huffman encoded */
			    for(x=1;x<=Hu2Sizes[HufN];x++)
				  {
				  lastX += Histogram[x] * ((long)SizeNum(x) - HufStruct->ln[x]);
				  }			  //data length change

			    MaxHuTbSize = PackSizes(NULL,HufStruct->ln,Hu2Sizes[HufN]);
			    lastX -= MaxHuTbSize;  //- length of the table decreases criteria

				// when a new bitstream is longer by more than 18 bits, don't try next splitting
			    if( (lastX<HufMax+18) && (Hu2Sizes[HufN]>32) )
					  break;     // function gets decreasing
			    if(lastX>HufMax)
				  {
				  HufMax=lastX;
				  y=HufN;
				  }
			    }
			   }

			else {  /* When no optimisation, just use one fixed splitting */
			     y=1;
			     HufMax=5;
			     }

			HufN=y;		// Optimal split is stored inside 'Huffn'
			Histogram[0]=HufZero;
			for(x=Hu2Sizes[HufN]+1;x<MAX_CODES;x++) Histogram[0]+=Histogram[x];

			memset(HufStruct->ln,0,MAX_CODES);	// HufStruct->ln contains uninitialised values.
			x = sq_huffman(Histogram,HufStruct->ln,BcountLen,Hu2Sizes[HufN]+1,ch_tab);
#ifdef _DEBUG
//			FILE *F = fopen("o:\\temp\\44\\huff.txt","wt");
//			for(int ii=0;ii<=Hu2Sizes[HufN];ii++) fprintf(F,"%d: len:%d\n", ii,HufStruct->ln[ii]);
//			fclose(F);
#endif

			  /* perform a codespace check */
			if(!HufCount(HufStruct, Hu2Sizes[HufN]+1))
			{     //WRONG CODESPACE - BIG SHIT !!!!
			  fprintf(stderr,"Invalid Huffman codespace, plane:%d! Internal workaround.\n",i);
			  isHuffman = 0;
			  goto NoHuff;
			}

			sq_wrhufi(HufStruct,BcountLen,Hu2Sizes[HufN]+1);	// Generate Hufman codes.

/*
			{FILE *F=fopen("o:\\temp\\44\\huff_.txt","wt");
		        if(F) {
			  for(int ii=0;ii<=Hu2Sizes[HufN];ii++) fprintf(F,"%d: len:%d cod:%Xh\n",ii,HufStruct->ln[ii],
			     swap_bits_order(HufStruct->cod[ii],HufStruct->ln[ii]) );
			  fclose(F);}}
*/
			isHuffman = HufMax>4;
			if(isHuffman)
				{
				if(PackSizes(NULL,HufStruct->ln,Hu2Sizes[HufN])==0) isHuffman=0;  //tohle zatim neumim
				}
			}

NoHuff:		sq_wrn(&BitStream,isHuffman,1);

		if(isHuffman)
			{
			sq_wrn(&BitStream,HufN,3);
				//Store Huffman Table
			PackSizes(&BitStream,HufStruct->ln,Hu2Sizes[HufN]);
			}
//-------end of Huffman compression------------}


		 lastX=-1;

		 for(y=0; y<PlaneX->Size2D; y++)
		 {
		   Rotm = 128;
		   pb = (uint8_t *)PlaneX->GetRowRaster(y)->Data1D;

		   for(x=0; x<PlaneX->GetSize1D(); x++)
	           {
		     if((*pb & Rotm) != 0)
		     {
			if(isHuffman) WriteHufNum(&BitStream,HufStruct,(long)(x)-lastX,HufN);
			         else WriteNumber(&BitStream,(long)(x)-lastX);
			lastX=x;
		     }
		     Rotm = Rotm >> 1;
		     if(Rotm==0)
		     {
		       Rotm=128;
		       pb++;
		     }
		   }
		   lastX = lastX - PlaneX->GetSize1D();
		 }
		 if(lastX<-1)
		 {
		   if(isHuffman) WriteHufNum(&BitStream,HufStruct,0-lastX,HufN);
			    else WriteNumber(&BitStream,0-lastX);
		 }

	       }
		 else {
		      sq_wrn(&BitStream,0,1);
		      for(y=0;y<PlaneX->Size2D;y++)
			   {
			   x=(PlaneX->GetSize1D()-1)>>3;
			   ((uint8_t *)(PlaneX->GetRowRaster(y)->Data1D))[x]=swap_bits_xlat[((uint8_t *)(PlaneX->GetRowRaster(y)->Data1D))[x]];
			   sq_wrUnl(&BitStream,(uint8_t *)PlaneX->GetRowRaster(y)->Data1D,PlaneX->GetSize1D());
			   ((uint8_t *)(PlaneX->GetRowRaster(y)->Data1D))[x]=swap_bits_xlat[((uint8_t *)(PlaneX->GetRowRaster(y)->Data1D))[x]];
			   }
		      }

	     }

	 TotalFlush(&BitStream);

	 free(CompBuffer); CompBuffer=NULL;
         if(PlaneX)
           {delete PlaneX;PlaneX=NULL;}
       }

   if(ObjLeader.size==~1)
   {
     ObjLeader.size = ftell(f) - StartPos - SizeImgObject;
     fseek(f, StartPos, SEEK_SET);
     if(SaveFtgObject(f,ObjLeader)!=SizeImgObject) {SavePictureFTG_=ErrWriteFile;break;} //error - cannot write object descriptor
     fseek(f, StartPos+ObjLeader.size+SizeImgObject, SEEK_SET);
   }

   if(CurrImg->Palette!=NULL)		// palete store
   {
     ObjLeader.type = 20;		//=palette;
     ObjLeader.size = CurrImg->Palette->Size1D*3 + sizeof(PaletteData);
     if(SaveFtgObject(f,ObjLeader)!=SizeImgObject) {SavePictureFTG_=ErrWriteFile;break;} //error - cannot write object descriptor

     PaletteDesc.items = CurrImg->Palette->Size1D;
     PaletteDesc.typ = 3;
     PaletteDesc.startItem = 0;
     PaletteDesc.destPlanes = CurrImg->Raster->GetPlanes();
     if(fwrite(&PaletteDesc,sizeof(PaletteData),1,f)!=1) {SavePictureFTG_=ErrWriteFile;break;}

     for(x=0; x<CurrImg->Palette->Size1D; x++)
     {
       fputc(CurrImg->Palette->R(x),f);	//Red;
       fputc(CurrImg->Palette->G(x),f);	//Green;
       fputc(CurrImg->Palette->B(x),f);	//Blue;
     }
   }

   if(!CurrImg->Properties.isEmpty())
   {
     unsigned PropPos = 0;
     while((Prop=CurrImg->Properties.Find("comment",PropPos)) != NULL)
     {
       ObjLeader.type = 1;		// Text comment
       ObjLeader.size = Prop->DataSize;
       if(SaveFtgObject(f,ObjLeader)!=SizeImgObject) {SavePictureFTG_=-12;break;} //error - cannot write object descriptor
       fwrite(Prop->Data,1,Prop->DataSize,f);
     }
     PropPos = 0;
     while((Prop=CurrImg->Properties.Find(ExifPreffix,PropPos)) != NULL)
     {
       ObjLeader.type = 40;		// EXIF
       ObjLeader.size = Prop->DataSize;
       if(SaveFtgObject(f,ObjLeader)!=SizeImgObject) {SavePictureFTG_=-12;break;} //error - cannot write object descriptor
       fwrite(Prop->Data,1,Prop->DataSize,f);
     }
     PropPos = 0;
     while((Prop=CurrImg->Properties.Find("XMP",PropPos)) != NULL)
     {
       ObjLeader.type = 41;
       ObjLeader.size = Prop->DataSize;
       if(SaveFtgObject(f,ObjLeader)!=SizeImgObject) {SavePictureFTG_=-12;break;} //error - cannot write object descriptor
       fwrite(Prop->Data,1,Prop->DataSize,f);
     }
     PropPos = 0;
     while((Prop=CurrImg->Properties.Find("ICCP",PropPos)) != NULL)
     {
       ObjLeader.type = 42;
       ObjLeader.size = Prop->DataSize;
       if(SaveFtgObject(f,ObjLeader)!=SizeImgObject) {SavePictureFTG_=-12;break;} //error - cannot write object descriptor
       fwrite(Prop->Data,1,Prop->DataSize,f);
     }
   }

   CurrImg = CurrImg->Next;
 }

 if(Histogram!=NULL) free(Histogram);
 if(HufStruct!=NULL) free(HufStruct);
 if(ch_tab!=NULL)    free(ch_tab);
 if(f!=NULL)	     fclose(f);

return(SavePictureFTG_);
}
#endif


#endif
//-------------------End of FTG routines------------------

#endif

