/** @file Plotter.h Prototypes for one plot inside a chart. */
#ifndef _PLOTTER_H
#define _PLOTTER_H

#include <vector>
#include <wx/pen.h>

#include "shape.h"


/**
*  The Plotter class which draws curve(s) in the window.
* 
*  This class takes the data needed for plotting one or several graph lines from a model. 
*  The graph(s) are plotted on the same set of x-y axes and displayed in a window. 
*  The graph is resized when the window is resized, maximized when the window is maximized and 
*  restored. */
class Plotter: public wxPanel
{
public:    
    Plotter(wxWindow *parent, const wxString &title, int LineNumber=3); ///< Constructor. Holds the window title and number of graph lines.
    virtual ~Plotter();			      ///< Destructor

    void SetXrange(double *xAxis);    
    void GetXrange(double *xAxisRange) const; ///< Gets the graphs' x-axis extreme ranges
    
    void SetYrange(double *yAxis, double *yCanvas=NULL); ///< Sets the graphs' y-axis extreme ranges    
    void GetYrange(double *yAxisRange) const; ///< Gets the graphs' y-axis extreme ranges
    
    void SetTitle(const wxString &NewTitle);	  ///<Sets the graph's title. Default name is "My Plot"    
    void AddPoint(int count, const double *Values, bool Paint=true); ///< Adds a point thanks to the signal number.
    //void OnMouseMove(wxMouseEvent &event);  ///< Shows the mouse position
    
    void SetBackground(const wxColour &color);///< Sets background color.
    void SetSignalLine(int lineNumber, const wxPen &pen); ///< Sets the properties of signal line(s).
    void ShowSignalLine(int lineNumber, bool Visible);
    void SetSwapLine(const wxPen &pen);	      ///< Sets swap line appearence.
    void ShowSwapLine(bool Visible);
    void SetAxesLine(const wxPen &pen);	      ///< Sets axes line appearence.
    void SetGridLine(const wxPen &pen);       ///< Sets grid line appearence.
    
    void SetTextFont(const wxFont &font);
    
    //void InitAllProp();    ///< was intended to initialise colors. Not used.        
        
    virtual void OnPaint(wxPaintEvent& event);///< flicker free repaint of whole chart
    void PaintMe(void);			      ///< Repaint a chart from code. OnPaint must not be used.
    void Clear(void);			      ///< erase data from all series

    void CalcTransforms(wxDC &dc);        
    bool dirty;				      ///< flag that signals neccessity to repaint

    void OnEraseBackground(wxEraseEvent& event);///< Does nothing to avoid flashing

    int GetPaintMode(void) const {return PaintMode;}; ///< PaintMode cannot be accessed directly, read it.
    void SetPaintMode(int NewMode);	      ///< PaintMode cannot be accessed directly, write it.    
    void SerVisible(int Index, bool Value);
    void SetXTicks(int NewTicks)	{GrX.Ticks=NewTicks;}    
    void SetYTicks(int NewTicks)	{GrY.Ticks=NewTicks;}
    void SetXLblTicks(int NewTicks)	{LblX.ticks=NewTicks;}    
    void SetYLblTicks(int NewTicks)	{LblY.ticks=NewTicks;}

protected:
    int PaintMode;			///< Style of painting chart, see PAINT_MODES.
    int maxGraphLineNumber;		///< Maximum graph lines exist
    int LEFT_GAP;	                ///< Shift the rectangle from the panel.
    int RIGHT_GAP;	                ///< Shift the rectangle from the panel.

    DECLARE_EVENT_TABLE()

    Transform2D Tx;			///< Main coordinate system of a chart.
    Transform2D TxAux1, TxAux2;		///< Auxiliary coordinate systems, when chart is split.
    XAxis AxX;				///< X axis object
    YAxis AxY;				///< Y axis object
    XLabels LblX;			///< X labels
    YLabels LblY;			///< Y labels
    HGrid GrY;				///< Vertical grid
    VGrid GrX;				///< Horizontal grid
    VCursor CurAct;			///< Splitter vertical line
    DataBuffer time;			///< Time data are common for all series
    DataBuffer *SerData;		///< Measured data for all series.
    Serie *Sr;				///< series to paint, they should be initialised by SerData
    
    wxString title;		        ///< Holds the title of the graph    

    wxColor backgroundColor;
    void DrawGraph(wxDC &dc);                 ///< Draw all chart stuff like axes, grid an so on on canvas.    
    void DrawSignals(wxDC &dc);               ///< Draws the lines from the time base to current time unless graph is shown fully.

private:
    /** The color of the background. The color of background is 
    *  used for painting the background of the panel's rectangle. */
    
        
    void ApplySignalLineProp(wxDC &devContext, int lineNumber); ///< Apply the properties of the signals line do DC.
    void SetSwapLineProp(wxDC &devContext);   ///< Sets the properties of the swap line.    
    void DrawPieceSignals(wxDC &dc);          ///< Incremental draw of new data.   
       
    wxFont textFont;  ///< An appearance of text. The font which determines the appearance of all the text in the panel.    
    void SetTextFontProp(wxDC &devContext);   ///< Sets the appearance of the text. Sets the properties of the text which is seen on the panel.
        
    const wxPen DEFAULT_AXES_LINE ;  ///< Default axes lines value        
    const wxColour DEFAULT_BGROUND;  ///< Default background colour value    
    const wxPen DEFAULT_GRID_LINE ;  ///< Default grid lines value
    const wxPen DEFAULT_SIGNAL_LINE; ///< Default signal lines value
    const wxPen DEFAULT_SWAP_LINE ;  ///< Default swap lines value        
    const wxFont DEFAULT_TEXT_FONT;  ///< Default text font value

    enum PAINT_MODES
     {
     FIXED = 0,			///< Chart has a fixed range and splitter line. Only minimal repaint needed.
     FLOATING = 1,		///< Chart has actual values on right axis and history goes to left.
     FIXED_F_REPAINT = 2
     };
          
};

#endif //_PLOTTER_H
